<?php
/**
 * E-Ticaret API Client
 * Cartelwind API entegrasyonu için client sınıfı
 */

require_once 'api_config.php';

class ECommerceAPIClient {
    private $baseUrl;
    private $apiKey;
    private $apiSecret;
    private $timeout;
    private $token = null;
    
    public function __construct() {
        $this->baseUrl = API_BASE_URL;
        $this->apiKey = API_KEY;
        $this->apiSecret = API_SECRET;
        $this->timeout = API_TIMEOUT;
    }
    
    /**
     * API Authentication
     * @return bool
     */
    private function authenticate() {
        if (!empty($this->token)) {
            return true;
        }
        
        try {
            $response = $this->makeRequest('auth', [
                'apikey' => $this->apiKey,
                'apisecret' => $this->apiSecret
            ], 'POST');
            
            if (isset($response['code']) && $response['code'] === 'SUCCESS') {
                $this->token = $response['result']['token'] ?? null;
                return !empty($this->token);
            }
            
            return false;
        } catch (Exception $e) {
            $this->logError('Authentication failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Sipariş oluştur
     * @param array $orderData Veritabanından gelen sipariş verisi
     * @return array
     */
    public function createOrder($orderData) {
        if (!$this->authenticate()) {
            return [
                'success' => false,
                'message' => 'API authentication failed'
            ];
        }
        
        // Sipariş verisini API formatına dönüştür
        $apiOrderData = $this->convertOrderToAPIFormat($orderData);
        
        try {
            $response = $this->makeRequest('order/create', $apiOrderData, 'POST', true);
            
            if (isset($response['code']) && $response['code'] === 'SUCCESS') {
                return [
                    'success' => true,
                    'message' => 'Sipariş başarıyla oluşturuldu',
                    'order_code' => $response['result']['orderCode'] ?? null,
                    'order_id' => $response['result']['id'] ?? null,
                    'response' => $response
                ];
            } else {
                return [
                    'success' => false,
                    'message' => $response['message'] ?? 'Sipariş oluşturulamadı',
                    'response' => $response
                ];
            }
        } catch (Exception $e) {
            $this->logError('Order creation failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'API request failed: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Veritabanı sipariş formatını API formatına dönüştür
     * @param array $dbOrder Veritabanından gelen sipariş
     * @return array
     */
    private function convertOrderToAPIFormat($dbOrder) {
        // API dokümantasyonuna göre sipariş formatı
        // Not: Dokümantasyondaki tam format bilinmediği için genel bir yapı oluşturuyoruz
        // Gerçek formatı API dokümantasyonundan kontrol etmeniz gerekebilir
        
        $apiData = [
            // Müşteri bilgileri
            'customer' => [
                'name' => $dbOrder['customer_name1'],
                'surname' => $dbOrder['customer_surname1'],
                'phone' => $dbOrder['customer_phone1'],
                'tcNo' => $dbOrder['customer_tckn'] ?? null,
            ],
            
            // Teslimat adresi
            'deliveryAddress' => [
                'city' => $dbOrder['delivery_city_name'],
                'district' => $dbOrder['delivery_district_name'],
                'neighbourhood' => $dbOrder['delivery_neighbourhood_name'],
                'address' => $dbOrder['delivery_address1'],
                'address2' => $dbOrder['delivery_address2'] ?? '',
            ],
            
            // Ürün bilgileri
            'items' => [
                [
                    'productCode' => $dbOrder['company_product_code'],
                    'quantity' => (int)$dbOrder['order_total_quantity'],
                    'price' => (float)$dbOrder['unit_price'],
                    'name' => $dbOrder['company_product_name'],
                ]
            ],
            
            // Sipariş bilgileri
            'orderCode' => $dbOrder['order_code'],
            'totalPrice' => (float)$dbOrder['total_price'],
            'taxRate' => (float)$dbOrder['tax_rate'],
            'paymentType' => $dbOrder['payment_type'] ?? 'cash',
            
            // Notlar
            'notes' => 'BIM Siparişi - Depo: ' . $dbOrder['bim_warehouse_name'] . ' - Mağaza: ' . $dbOrder['bim_store_name'],
        ];
        
        // İkinci müşteri varsa ekle
        if (!empty($dbOrder['customer_name2'])) {
            $apiData['customer2'] = [
                'name' => $dbOrder['customer_name2'],
                'surname' => $dbOrder['customer_surname2'],
                'phone' => $dbOrder['customer_phone2'],
            ];
        }
        
        return $apiData;
    }
    
    /**
     * API isteği yap
     * @param string $endpoint
     * @param array $data
     * @param string $method
     * @param bool $requiresAuth
     * @return array
     */
    private function makeRequest($endpoint, $data = [], $method = 'GET', $requiresAuth = false) {
        $url = rtrim($this->baseUrl, '/') . '/' . ltrim($endpoint, '/');
        
        $ch = curl_init();
        
        $headers = [
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        if ($requiresAuth && !empty($this->token)) {
            $headers[] = 'Authorization: Bearer ' . $this->token;
        }
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data, JSON_UNESCAPED_UNICODE));
        } elseif ($method === 'GET' && !empty($data)) {
            $url .= '?' . http_build_query($data);
            curl_setopt($ch, CURLOPT_URL, $url);
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception("cURL Error: " . $error);
        }
        
        $decodedResponse = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid JSON response: " . $response);
        }
        
        if (API_DEBUG) {
            $this->logRequest($endpoint, $data, $decodedResponse, $httpCode);
        }
        
        return $decodedResponse;
    }
    
    /**
     * Hata loglama
     */
    private function logError($message) {
        if (API_DEBUG) {
            error_log('[ECommerce API] ' . date('Y-m-d H:i:s') . ' - ' . $message);
        }
    }
    
    /**
     * İstek loglama
     */
    private function logRequest($endpoint, $requestData, $response, $httpCode) {
        $logData = [
            'timestamp' => date('Y-m-d H:i:s'),
            'endpoint' => $endpoint,
            'request' => $requestData,
            'response' => $response,
            'http_code' => $httpCode
        ];
        
        $logFile = __DIR__ . '/api_logs/' . date('Y-m-d') . '.log';
        $logDir = dirname($logFile);
        
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        file_put_contents($logFile, json_encode($logData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n\n", FILE_APPEND);
    }
}

